<?php
if (session_status() !== PHP_SESSION_ACTIVE) session_start();
include 'conexao.php';

// Verifica se o usuário é um professor autenticado
if (!isset($_SESSION['usuario']) || strtoupper($_SESSION['usuario']['perfil']) !== 'PROFESSOR') {
    header("Location: login.php");
    exit;
}

$prof_id = $_SESSION['usuario']['id'];
$erros = [];
$sucesso = "";

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $titulo = $_POST['titulo'] ?? '';
    $descricao = $_POST['descricao'] ?? '';
    $turma_id = $_POST['turma_id'] ?? '';
    $disciplina_id = $_POST['disciplina_id'] ?? '';
    $anexo = '';

    if (!$titulo || !$descricao || !$turma_id || !$disciplina_id) {
        $erros[] = "Todos os campos são obrigatórios.";
    } else {
        // upload do anexo
        if (!empty($_FILES['anexo']['name'])) {
            $ext = pathinfo($_FILES['anexo']['name'], PATHINFO_EXTENSION);
            $nome_arquivo = uniqid() . '.' . $ext;
            // Garante que o diretório de upload existe
            if (!is_dir('uploads_trabalhos')) {
                mkdir('uploads_trabalhos', 0777, true);
            }
            move_uploaded_file($_FILES['anexo']['tmp_name'], 'uploads_trabalhos/' . $nome_arquivo);
            $anexo = $nome_arquivo;
        }

        $stmt = $conn->prepare("INSERT INTO trabalhos (titulo, descricao, data_envio, disciplina_id, turma_id, professor_id, anexo) VALUES (?, ?, NOW(), ?, ?, ?, ?)");
        $stmt->bind_param("ssiiss", $titulo, $descricao, $disciplina_id, $turma_id, $prof_id, $anexo);
        if ($stmt->execute()) {
            $sucesso = "Trabalho publicado com sucesso!";
        } else {
            $erros[] = "Erro ao salvar o trabalho: " . $stmt->error;
        }
    }
}

// Buscar apenas as turmas e disciplinas do professor logado, usando a tabela 'atribuicoes'
$turmas_query = "SELECT DISTINCT t.id, t.nome, t.classe, t.curso FROM turmas t 
                 JOIN atribuicoes a ON t.id = a.turma_id
                 WHERE a.professor_id = ?";
$stmt_turmas = $conn->prepare($turmas_query);
$stmt_turmas->bind_param("i", $prof_id);
$stmt_turmas->execute();
$turmas = $stmt_turmas->get_result();

// Disciplinas do professor logado, para a primeira turma selecionada (se houver)
$disciplinas_query = "SELECT d.id, d.nome FROM disciplinas d
                      JOIN atribuicoes a ON d.id = a.disciplina_id
                      WHERE a.professor_id = ? AND a.turma_id = ?";
$stmt_disciplinas = $conn->prepare($disciplinas_query);

?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <title>Cadastrar Trabalho</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body { font-family: Arial, sans-serif; background: #f5f5f5; padding: 20px; color: #333; }
        .container { max-width: 600px; margin: auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h2 { text-align: center; color: #002244; }
        label { display: block; margin-top: 10px; }
        input[type="text"], textarea, select {
            width: 100%; padding: 8px; margin-top: 5px; border: 1px solid #ccc; border-radius: 4px;
        }
        button {
            margin-top: 15px;
            padding: 10px 15px;
            background: #002244;
            color: white;
            border: none;
            border-radius: 5px;
            cursor: pointer;
        }
        .msg { margin-top: 10px; text-align: center; font-weight: bold; }
        .erro { color: red; }
        .sucesso { color: green; }
    </style>
    <script>
        function carregarDisciplinas(turma_id) {
            const disciplinaSelect = document.getElementById('disciplina');
            disciplinaSelect.innerHTML = '<option>Carregando...</option>';

            // A chamada para `disciplinas_por_turma.php` agora também inclui o professor_id
            // para carregar apenas as disciplinas relevantes
            fetch(`disciplinas_por_turma.php?turma_id=${turma_id}&professor_id=<?= $prof_id ?>`)
                .then(response => response.json())
                .then(data => {
                    disciplinaSelect.innerHTML = '<option value="">Selecione a disciplina</option>';
                    data.forEach(function(disc) {
                        const opt = document.createElement('option');
                        opt.value = disc.id;
                        opt.textContent = disc.nome;
                        disciplinaSelect.appendChild(opt);
                    });
                })
                .catch(error => {
                    console.error('Erro ao carregar disciplinas:', error);
                    disciplinaSelect.innerHTML = '<option value="">Erro ao carregar disciplinas</option>';
                });
        }
    
        window.onload = function() {
            // Carrega as disciplinas para a primeira turma na lista ao carregar a página
            const turmaSelect = document.getElementById('turma');
            if (turmaSelect.value) {
                carregarDisciplinas(turmaSelect.value);
            }
        };
    </script>
</head>
<body>

<div class="container">
    <h2>Publicar Trabalho para a Turma</h2>

    <?php if ($erros): ?>
        <div class="msg erro"><?= implode('<br>', $erros) ?></div>
    <?php endif; ?>

    <?php if ($sucesso): ?>
        <div class="msg sucesso"><?= $sucesso ?></div>
    <?php endif; ?>

    <form method="post" enctype="multipart/form-data">
        <label for="turma">Turma:</label>
        <select name="turma_id" id="turma" onchange="carregarDisciplinas(this.value)" required>
            <option value="">Selecione a turma</option>
            <?php while ($t = $turmas->fetch_assoc()): ?>
                <option value="<?= $t['id'] ?>"><?= htmlspecialchars($t['classe'] . '-' . $t['nome'] . '-' . $t['curso']) ?></option>
            <?php endwhile; ?>
        </select>

        <label for="disciplina">Disciplina:</label>
        <select name="disciplina_id" id="disciplina" required>
            <option value="">Selecione a disciplina</option>
        </select>

        <label for="titulo">Título do Trabalho:</label>
        <input type="text" name="titulo" required>

        <label for="descricao">Descrição:</label>
        <textarea name="descricao" rows="5" required></textarea>

        <label for="anexo">Anexo (opcional):</label>
        <input type="file" name="anexo" accept=".pdf,.doc,.docx,.jpg,.png">

        <button type="submit">Salvar Trabalho</button>
    </form>
</div>

</body>
</html>